import numpy as np
import math
import pylab as plt

# ************************************************************************
def Offset_Correction(Value, Velocity_Offset):
#
# Description:    Correct a binary velocity value if an offset is attached
# -----------     to it.
#
#                 Input: Value = binary data value read from file
#                        Velocity_Offset, binary value read from the data file
#                        Nb_Gates, binary value read from the data file

    Value = Value + Velocity_Offset
    ICOR = 0
    if Value > 127:
      ICOR = -256
    if Value < -128:
      ICOR = 256
    Value = Value + ICOR
    Value = float(Value - Velocity_Offset)

    return Value

# ************************************************************************
def Get_Dop_Profiles(fid, Graph, Channel, Block):
#
# Version: 5.0
# ------------
# UDOP version in file must be >= 3.00.1
#
#
# Description:    This function extracts from an UDOP binary data file all
# -----------     the data profiles contained in single graph for a specific
#                 channel and a specific block.
#
#                 The function checks if the handle associated to the data
#                 file points to a correct UDOP binary data file.
#                 The output variable Nb_Profiles an contains an error code
#                 if its value is < 0 or the number of available profile if > 0
#
# Input:          fid     : a handle to an already opened binary data file
#                 Graph   : the number of the curve contains in the profile [1,2....]
#                 Channel : Number of the channel
#                           In case of UDV 2D or 3D not relevant
#                 Block   : number of data block  [1,2....]
#
#
# Output:         X  : a 1 dimension data vector that gives the values of
#                      the abscissa for the selected graph. These
#                      values are depths in mm, but can be velocities in
#                      mm/s in case of the power spectrum of a single gate
#
#                 Y  : a 2 dimensions data table that contains the values of
#                      the measured data. Each row is associated to a profile
#                      and each column is associated to a gate or depth.
#
#                      In case of raw data acquisition only:
#                      row:           data
#                       1     first emission, I signal
#                       2     first emission, Q signal
#                       3     second emission, I signal
#                       4     second emission, Q signal
#                                     etc....
#
#                 Np : if > 0  number of available data profiles
#                              (= number of rows of the Y table,
#                               or number of emissions in case of
#                               raw data acquisition)
#                      if < 0  error code
#                          -1 : not an UDOP binary data file
#                          -2 : unsupported software version
#                          -3 : the selected graphs does not exists
#                          -4 : the selected channel does not exists
#                          -5 : the selected block does not exists
#
#                 Channel_Para : of table that contains all the value of the parameters
#                              associated to the selected channel
#
#                 time stamp : a vector that gives the time stamp in ms
#                              The length of the vector equal the number of rows
#                              of the Y table
#
# Structure of a binary profile
# -----------------------------
# A : 1 word that gives the total number of bytes contained in the profile
# B : 1 word that gives the number of bytes contained in the following graph
#     if 0 no more graph in the profile
# C : 1 byte that identify the type of data contained in the graph
#     (see table "Data_Type constant")
# D : the data of the graph
#     if each data are in byte format or word format depending the data_type
#
# Struture B,C,D repeat until B = 0
#
# E : 1 doubleword = time stamps in ms*10
# F : 1 word = number of the block
# G : 1 byte reserved for further use
# H : 1 byte reserved for further use
# I : 1 byte that gives the channel number
#
# J : 1 word that gives the total number of bytes contained in the profile
#     (J = A)
#
# Date: 25 december 2020
# Author : J.-Cl Willemetz, Signal Processing SA
#          For any questions or comments please email to:
#          contact@signal-processing.com
# -------------------------------------------------------------------------
    X = []
    Y = []
    Np = 0
    Parameters = []
    TBP = []

    fid.seek(-1,2)
    eof = fid.tell()
    fid.seek(0,0)
#
# Check if the file comes from UDOP software
#
    File_Ident = fid.read(7)
    S=File_Ident.decode('utf-8')
    if S != 'BINUDOP':
      if S != 'BIQUDOP':
        return X,Y,-1,Parameters,TBP

# Get the software version of file
#
    S = fid.read(1)
    V = fid.read(6)
    V =V.decode('utf-8')
    V1=int(V[0])
    V2=int(V[2:4])
    V3=int(V[5])
    Version = 1000*V1 + 10*V2 + V3;
    if Version < 3001:
      return X,Y,-2,Parameters,TBP
#
# Defermine the DOP model
# -----------------------
# The signature was not always correctly added to the BDD file for the DOP4000.
# Therfore for the DOP4000, if the number (first two digit) is zero, the option bytes are
# used to indentify the model.

    Dop_Model = 0

    fid.seek(528, 0)          # point to identification area
    Dop_Ident = fid.read(2)
    S = Dop_Ident.decode('utf-8')

    if S == '12':
      Dop_Model = 3000

    if S == '13':
      Dop_Model = 4000

    if S == '14':
      Dop_Model = 5000

    if Dop_Model == 0:
      fid.seek(538, 0)          # point to identification area
      Opt_0 = fid.read(1)
      Opt_1 = fid.read(1)
      Opt_2 = fid.read(1)
      if Opt_0 == 0:
        if Opt_1 == 0xFF:
          if (Opt_2 & 0x9) == 0x9:
                    DopModel = 4000

    if Dop_Model == 0:
       return X,Y,-1,Parameters,TBP
#

# Some UDOP constants -----------------------------------
    Nb_Para_Fct = 256                     # Number of bytes of the table of parameters associated to a channel
    Ofs_Para_Table = 548

    Data_Type_Velocity = 0
    Data_Type_Echo = 1
    Data_Type_Energy = 2
    Data_Type_Gate_FFT = 3
    Data_Type_Phase_deg_10 = 4           # phase in Deg*10, format 16 bits
    Data_Type_Vitson = 5
    Data_Type_Frequency = 6
    Data_Type_Frequency_TR1 = 7
    Data_Type_Frequency_TR2 = 8
    Data_Type_Frequency_TR3 = 9
    Data_Type_Echo_TR1 = 10
    Data_Type_Echo_TR2 = 11
    Data_Type_Echo_TR3 = 12
    Data_Type_Energy_TR1 = 13
    Data_Type_Energy_TR2 = 14
    Data_Type_Energy_TR3 = 15
    Data_Type_Velocity_mm_s_10 = 16       # Velocity in mm/s * 10
    Data_Type_Velocity_mm_s = 17          # Velocity in mm/s
    Data_Type_Flow_ml_min = 18            # Flowrate in ml/min,  format 32 bits
    Data_Type_Diameter_10 = 19            # diameter in mm*10, format 16 bits
    Data_Type_Aliasing_Refer = 20         # reference profil base on 2 PRF
    Data_Type_Aliasing_Refer_TR1 = 21     # reference profil base on 2 PRF for TR1
    Data_Type_Aliasing_Refer_TR2 = 22     # reference profil base on 2 PRF for TR2
    Data_Type_Aliasing_Refer_TR3 = 23     # reference profil base on 2 PRF for TR3
    Data_Type_TGC = 28
    Data_Type_IQ = 29
    Data_Type_Frequency_Hz_10 = 24
    Data_Type_Depth_mm_10 = 25            # depths [mm*10]
    Data_Type_Angle_Deg_10 = 30           # angle [deg*10], format 16 bits

# Variables which depend on DOP instrument
# ----------------------------------------
    if Dop_Model == 3000:
      Ofs_Data = 3620
      Channel_Max = 1

    if Dop_Model == 3010:
      Ofs_Data = 31268
      Channel_Max = 10

    if Dop_Model == 4000:
      Ofs_Data = 12836
      Channel_Max = 4

    if Dop_Model == 5000:
      Ofs_Data = 49700
      Channel_Max = 16

# Initialize some variables
# -------------------------
    Nb_Gates = 0
#
#
# Read the parameters for the selected channel
# --------------------------------------------
    if Channel > Channel_Max:
      return X,Y,info,-4,Channel_Parameters

    fid.seek(Ofs_Para_Table + 1024*(Channel-1), 0)
    Parameters = np.fromfile(fid, dtype=np.int32, count=Nb_Para_Fct, sep="")
#
    i = -1
    Freq_Emi = Parameters[i+1]     # Emitting frequency in kHZ
    Prf_Periode_us = Parameters[i+6] # PRF in micro seconds
    Indice_First_Gate = Parameters[i+10]
    Skip_Gate = Parameters[i+11] # time between gates indice
    Nb_Gates = Parameters[i+14] #
    Nprf = Parameters[i+15] #
    Scale_Angle = Parameters[i+16] # velocity scaling factor
    Sound_Speed = Parameters[i+20] # sound velocity in m/s
    Doppler_Angle = Parameters[i+21] # Doppler angle in degrees
    Module_Scale = Parameters[i+22]     # Emitting frequency in kHZ
    Velocity_Offset = Parameters[i+23] # Coded value of the velocity Offset
    No_Flt_ispPAC = Parameters[i+28]   # IQ filter indice
    Freq_AD_Info    = Parameters[i+30] # fixe the AD frequency
    FFT_Scale_Pmax = Parameters[i+31]
    Mpx_Mode = Parameters[i+53]
    Freq_Latch_Tgc_kHz = Parameters[i+41] / 1000.0
    UDOP_Fct_Mode =  Parameters[i+44]
    Cursor_On_Gate = Parameters[i+61]

    Tr1_Frequency_Offset = Parameters[i+71]
    Tr2_Frequency_Offset = Parameters[i+72]
    Tr3_Frequency_Offset = Parameters[i+73]
    MD_Scale_Angle_Tr1 = Parameters[i+74]
    MD_Scale_Angle_Tr2 = Parameters[i+75]
    MD_Scale_Angle_Tr3 = Parameters[i+76]
    MD_Module_Scale_Tr1 = Parameters[i+77]     # Emitting frequency in kHZ
    MD_Module_Scale_Tr2 = Parameters[i+78]     # Emitting frequency in kHZ
    MD_Module_Scale_Tr3 = Parameters[i+79]     # Emitting frequency in kHZ

    Acqui_IQ_Nprf = Parameters[i+91]
    Acqui_IQ_Nb_Gates = Parameters[i+92]
    Acqui_IQ_First_Gate = Parameters[i+93]

    Over_All_Gain_dB = Parameters[i+94]        # gain for FFT
    Flow_unit = Parameters[i+112]              # 0 : ml/min
                                                 # 1 : ml/s
                                                 # 2 : dl/s
    Flow_Scale = Parameters[i+113]
    Aliasing_Ctrl_1 = Parameters[i+114]
    Aliasing_Ctrl_2 = Parameters[i+115]

    Sound_Speed_Wall = Parameters[i+116]
    Length_Wall = Parameters[i+117]
    Sound_Speed_Couplant = Parameters[i+118]
    Length_Couplant = Parameters[i+119]
    Div_Freq_AD = Parameters[i+121]
    DOP4000_Mod_Fct1 = Parameters[i+122]
    DOP4000_Tgc_values = Parameters[i+123]
    Histo_Nb_Classes = Parameters[i+124]
    DOP5000_Mod_Fct1 = Parameters[i+125]
    Mpx5000_Mode_1 = Parameters[i+126]
    DOP5000_Tgc_Fct1 = Parameters[i+127]
    DOP5000_Group_Mode1 = Parameters[i+128]
    DOP5000_Group_Mode2 = Parameters[i+129]
    DOP5000_Group_Mode3 = Parameters[i+130]
    DOP5000_Group_Mode4 = Parameters[i+131]
    Nb_Gates_Desired = Parameters[i+132]


# -------------------------------------------------------------------------
#   Compute some useful variables
#
    Cos_AngDop = math.cos(Doppler_Angle*np.pi/180)
    Coded_Value_To_Frequency = 1000 * Scale_Angle/(256*np.pi*Prf_Periode_us)
    Coded_Value_To_Velocity = 128 * Coded_Value_To_Frequency * Sound_Speed /(256*Freq_Emi * Cos_AngDop)
    Coded_Value_To_Frequency_For_Tr1 = 1000 * MD_Scale_Angle_Tr1 /(256*np.pi*Prf_Periode_us)
    Coded_Value_To_Frequency_For_Tr2 = 1000 * MD_Scale_Angle_Tr2 /(256*np.pi*Prf_Periode_us)
    Coded_Value_To_Frequency_For_Tr3 = 1000 * MD_Scale_Angle_Tr3 /(256*np.pi*Prf_Periode_us)

# determine the number of pseudo profiles
# ---------------------------------------
#   The first profiles in a block contain the depths of the gates. These are pseudo profiles
#   as they do not contain measured data. It exists normaly one pseudo profile but in
#   case of sequential acquisition the number of pseudo profiles equal the number of
#   used channels in a block.
#
    Number_Of_Pseudo_Profiles = 1

    if Dop_Model == 5000:
      Group = ((Channel-1) >> 2) + 1
      if (DOP5000_Mod_Fct1 & 0xF) == 2:
        # sequential mode
        Number_Of_Pseudo_Profiles = 0
        imax = 0x10000
        i = 0x1
        while i != imax:
          if (Mpx5000_Mode_1 & i) != 0:
            Number_Of_Pseudo_Profiles = Number_Of_Pseudo_Profiles + 1
          i = i << 1
      if (DOP5000_Mod_Fct1 & 0xF) == 1:
        # simultaneous mode
        i = 126 + Group
        i = Parameters[i]   # i = DOP5000_Group_Mode of group
        Number_Of_Pseudo_Profiles = (i & 3) + 1

    else:
      if (Mpx_Mode & 2) != 0:
        # sequential mode
        Number_Of_Pseudo_Profiles = 0
        imax = 0x4000
        i = 0x8
        while i != imax:
          i = i << 1
          if (Mpx_Mode & i) != 0:
            Number_Of_Pseudo_Profiles = Number_Of_Pseudo_Profiles + 1

# -------------------------------------------------------------------------
# Get the measurement depths
#
    try:
      fid.seek(Ofs_Data, 0)   # point to the beginning of the data
      N = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]  # N = number of bytes in a profile
      fid.seek(N-5, 1)    # point to the channel value
      Channel_Red = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
      Channel_first = Channel_Red
      while Channel_Red != Channel:
        fid.seek(2, 1) # point to the next profile
        N = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]  # N = number of bytes in a profile
        fid.seek(N-5, 1)   # point to the block value
        Channel_Red = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
        if Channel_Red == Channel_first: # All the pseudo profiles have been red
          return X,Y,-4,Parameters,TBP
    except:
      return X,Y,-4,Parameters,TBP

    # Replace the file pointer at the beginning of the profile
    fid.seek(-(N-2), 1)

    Nb_Bytes_In_Profile = np.fromfile(fid, dtype=np.uint16, count=1, sep="")[0]
    Nb_Bytes_In_Graph = np.fromfile(fid, dtype=np.uint16, count=1, sep="")[0]
    Data_Type_Red = np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
    N = Nb_Bytes_In_Graph // 2

    # assume data_type is depths values and place the values in millimeters
    for i in range (0, N):
      Vabs = 0.1 * np.fromfile(fid, dtype=np.uint16, count=1, sep="")[0]
      X.append(Vabs)

    if Data_Type_Red == Data_Type_Velocity_mm_s:     # in case of gate_FFT
      # X values are velocity values
      for i in range (0, N):
        X[i] = X[i] * 10;

    if Data_Type_Red == Data_Type_Diameter_10:
      # X is a single value, the  diameter of the pipe
      fid.seek(-Nb_Bytes_In_Graph, 1)
      X = []
      Vabs = 0.1 * np.fromfile(fid, dtype=np.int32, count=1, sep="")[0]
      X.append(Vabs)

#   Point to the first data profile
# ---------------------------------
    fid.seek(Ofs_Data, 0)   # point to the beginning of the selected block
    i = 0
    Ofs_Profile_Start = Ofs_Data
    while i != Number_Of_Pseudo_Profiles:
      N = np.fromfile(fid, dtype=np.uint16, count=1, sep="")[0]
      fid.seek(N-2, 1)
      i = i + 1
      Ofs_Profile_Start = Ofs_Profile_Start + N

# Point to the selected Block
# ---------------------------
    try:
      fid.seek(Ofs_Profile_Start, 0)
      N = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]  # N = number of bytes in a profile
      fid.seek(N-10, 1)   # point to the block value
      No_Block_Red = np.fromfile(fid, dtype=np.uint16, count=1, sep="")[0]

      while No_Block_Red != Block:
        fid.seek(6, 1) # point to the next profile
        Ofs_Profile_Start = Ofs_Profile_Start + N
        N = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]  # N = number of bytes in a profile
        fid.seek(N-10, 1)   # point to the block value
        No_Block_Red = np.fromfile(fid, dtype=np.uint16, count=1, sep="")[0]

    except:
      return X,Y,-5,Parameters,TBP

     # Now, Ofs_Profile_Start point the first profile of the selected block

#   Point to the first profile corresponding to the selected channel
#   ----------------------------------------------------------------
    fid.seek(Ofs_Profile_Start, 0)           # point to the beginning of the data profiles
    Ofs_Channel_Start = Ofs_Profile_Start
    N = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]  # N = number of bytes in a profile
    fid.seek(N-5, 1)   # point to the block value
    Channel_Red = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
    while Channel_Red != Channel:
      fid.seek(2, 1) # point to the next profile
      Ofs_Channel_Start = Ofs_Channel_Start + N
      N = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]  # N = number of bytes in a profile
      fid.seek(N-5, 1)   # point to the channel value
      Channel_Red = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]

    # Replace the file pointer at the beginning of the profile
    fid.seek(-(N-2), 1)

#   Determine the offset in the profile for the selected graph
#   ----------------------------------------------------------
    fid.seek(Ofs_Channel_Start + 2, 0) # point to the number of values in the graph
    Nb_Bytes_In_Graph = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]  # N = number of bytes in graph
    Data_Type = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
    Ofs_Graph = 5
    Graph_Cnt = 1
    while Graph_Cnt != Graph:
      fid.seek(Nb_Bytes_In_Graph, 1) # point to the number of values in the graph
      Nb_Bytes_In_Graph = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]  # N = number of bytes in graph
      if Nb_Bytes_In_Graph == 0:
        return X,Y,-3,Parameters,TBP
      Ofs_Graph = Ofs_Graph + Nb_Bytes_In_Graph + 3
      Data_Type = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
      Graph_Cnt = Graph_Cnt + 1

#   Extract the data values ....
#   -----------------------------
#   Ofs_Channel_Start point to the first profile to be extracted
#   and Ofs_Graph give the position inside the profile of the selected data type

    Stop_Extraction = 0
    Np = 0

    while Stop_Extraction == 0:

      fid.seek(Ofs_Channel_Start, 0)
      N = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]
      fid.seek(Ofs_Channel_Start + N- 12, 0)
      Time_Stamp = np.fromfile(fid, dtype=np.uint32, count=1, sep="")[0] # read time stamp of the profile
      fid.seek(Ofs_Channel_Start + Ofs_Graph, 0)

      if Data_Type == Data_Type_IQ:
        Nb_Gates = Int32(Nb_Bytes_In_Graph/4)
        for i in range(0, 2*Nb_Gates):
          Value = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]
          Y.append(Value)
          Value = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]
          Y.append(Value)
        Np = Acqui_IQ_Nprf
        break

      if Data_Type == Data_Type_Velocity:
        Nb_Gates = Nb_Bytes_In_Graph
        for i in range(0, Nb_Gates):
          Value = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
          Value = Offset_Correction(Value,Velocity_Offset) * Coded_Value_To_Velocity
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Echo:
        Nb_Gates = Nb_Bytes_In_Graph
        m = Module_Scale / 256
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Energy:
        Nb_Gates = Nb_Bytes_In_Graph
        m = Module_Scale / 256
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Frequency_TR1:
        Nb_Gates = Nb_Bytes_In_Graph
        for i in range(0, Nb_Gates):
          Value = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
          Value = Value * Coded_Value_To_Frequency_For_Tr1
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Frequency_TR2:
        Nb_Gates = Nb_Bytes_In_Graph
        for i in range(0, Nb_Gates):
          Value = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
          Value = Value * Coded_Value_To_Frequency_For_Tr2
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Frequency_TR3:
        Nb_Gates = Nb_Bytes_In_Graph
        for i in range(0, Nb_Gates):
          Value = np.fromfile(fid, dtype=np.int8, count=1, sep="")[0]
          Value = Value * Coded_Value_To_Frequency_For_Tr3
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Echo_TR1:
        Nb_Gates = Nb_Bytes_In_Graph
        m = MD_Module_Scale_Tr1 / 256
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Echo_TR2:
        Nb_Gates = Nb_Bytes_In_Graph
        m = MD_Module_Scale_Tr2 / 256
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Echo_TR3:
        Nb_Gates = Nb_Bytes_In_Graph
        m = MD_Module_Scale_Tr3 / 256
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Energy_TR1:
        Nb_Gates = Nb_Bytes_In_Graph
        m = MD_Module_Scale_Tr1 / 256
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Energy_TR2:
        Nb_Gates = Nb_Bytes_In_Graph
        m = MD_Module_Scale_Tr2 / 256
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Energy_TR3:
        Nb_Gates = Nb_Bytes_In_Graph
        m = MD_Module_Scale_Tr3 / 256
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Gate_FFT:
        Nb_Gates = Nb_Bytes_In_Graph
        m = (FFT_Scale_Pmax - FFT_Scale_Pmin)/255.0 + FFT_Scale_Pmin
        for i in range(0, Nb_Gates):
          Value = m * np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
          Y.append(Value - Over_All_Gain_dB)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Velocity_mm_s_10:
        Nb_Gates = Nb_Bytes_In_Graph / 2
        for i in range(0, Nb_Gates):
          Value = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]
          Y.append(Value/10)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Velocity_mm_s:
        Nb_Gates = Nb_Bytes_In_Graph / 2
        for i in range(0, Nb_Gates):
          Value = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]
          Y.append(Value)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Angle_Deg_10:
        Nb_Gates = Nb_Bytes_In_Graph / 2
        for i in range(0, Nb_Gates):
          Value = np.fromfile(fid, dtype=np.int16, count=1, sep="")[0]
          Y.append(Value/10)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Frequency_Hz_10:
        Nb_Gates = Nb_Bytes_In_Graph / 2
        for i in range(0, Nb_Gates):
          Value = np.fromfile(fid, dtype=np.uint16, count=1, sep="")[0]
          Y.append(Value/10)
        TBP.append(Time_Stamp)

      if Data_Type == Data_Type_Flow_ml_min:
        Nb_Gates = 1
        Value = np.fromfile(fid, dtype=np.uint32, count=1, sep="")[0]
        Y.append(Value)
        TBP.append(Time_Stamp)

      # point to the next profile
      Np = Np + 1

      Stop_Extraction = 0
      Ofs_Channel_Start = Ofs_Channel_Start + N
      fid.seek(Ofs_Channel_Start + N - 12, 0)
      if fid.tell() >= eof:
        Stop_Extraction = 1
      else:
        # The time stamp identifies the next profile
        Next_Time_Stamp = np.fromfile(fid, dtype=np.uint32, count=1, sep="")[0]
        while Next_Time_Stamp == Time_Stamp:
          Ofs_Channel_Start = Ofs_Channel_Start + N
          fid.seek(Ofs_Channel_Start + N - 12, 0)
          if fid.tell() >= eof:
            Stop_Extraction = 1
            break
          Next_Time_Stamp = np.fromfile(fid, dtype=np.uint32, count=1, sep="")[0]
        fid.seek(Ofs_Channel_Start + N-8, 0)
        # remain in the same block and lock for the same channel
        No_Bl = np.fromfile(fid, dtype=np.uint16, count=1, sep="")[0]
        fid.seek(3, 1)
        No_Ch = np.fromfile(fid, dtype=np.uint8, count=1, sep="")[0]
        if No_Bl != Block:
          Stop_Extraction = 1
        if No_Ch != Channel:
          Stop_Extraction = 1


#--------------------------------------------------


    Y = np.reshape(Y,(Np, Nb_Gates))
    TBP = np.reshape(TBP,(Np))
    return X,Y,Np,Parameters,TBP

#*************************************************************************

path="d:/signal/dop5000/data_trav/"

# Example 1
# ---------
# UDOP data file contains 1 graph of velocity profile from channel 1
# Extract all the profiles from the first block, compute the mean profile
# and plot the result.
#
Channel = 1
Block = 1
Graph = 2

My_File = path +'dop_data.bdd'
try:
  fid = open(My_File, 'rb');
except:
  print("The file", filename, " is not found")

# Read from file all the measured data of graph 1
#
[X, Y, Nb_Profiles, Channel_Parameters, TBP] = Get_Dop_Profiles(fid, Graph, Channel, Block)
#
# Check for some errors ....
if Nb_Profiles < 0 :
  if Nb_Profiles == -1:
    print('Not a comptabilble DOP data file')
  if Nb_Profiles == -2:
    print('Data file version not supported')
  if Nb_Profiles == -3:
    print('Unknown graph number')
  if Nb_Profiles == -4:
    print('Unknown channel')
  if Nb_Profiles == -5:
    print('Unknown block')
else:
  # Plot the mean profile
  # ---------------------
  Yp = np.mean(Y,axis=0)

  plt.figure()
  plt.plot(X, Yp)
  print('Number of profiles = ', Nb_Profiles)
  plt.show()

