% ************************************************************************
function [Parameters, Error] = Get_Para_Table(fid, Channel)
%
% Version: 4.0
% ------------
% from UDOP version 4.00.1
%
%
% Description:    Read all the parameters for the selected channel
% -----------    
%
%                 UDOP version supported: From 2.00.1
%
% Input:          fid   : a handle to an already opened binary data file
%                 Channel: Number of the channel
%                          for DOP3000 must be set to 1
%
%
% Output:         X  : a 1 dimension table that contains the parameters
%                      the X abscissa for the selected graph. These
%                      values are depths in mm, but can be velocities in
%                      mm/s in case of the power spectrum of a single gate
%
%               Error = 0 No error
%                      if < 0  error code
%                          -1 : not an UDOP binary data file
%                          -2 : unsupported software version
%                          -3 : the selected graphs does not exists
%                          -4 : the selected channel does not exists
%                          -5 : the selected block does not exists
%
%
% Date: 11 July 2016
% Author : J.-Cl Willemetz, Signal Processing SA
%          For any questions or comments please email to:
%          contact@signal-processing.com
% -------------------------------------------------------------------------
       frewind(fid);
%
% Check if the file comes from UDOP software
%
       S = fread(fid, 7,'*char');
       S = transpose(S);
       if strcmp(S, 'BINUDOP') == 0
         if strcmp(S, 'BIQUDOP') == 0
           Error = -1;
           return;
         end
       end  
%       
% Get the software version of file
%
       S = fread(fid, 1,'uint8');
       V = fread(fid, 6,'char');
       V1 = str2num(char(V(1)));
       V2 = 10*str2num(char(V(3))) + str2num(char(V(4)));      
       V3 = str2num(char(V(6)));
       Version = 1000*V1 + 10*V2 + V3;    
       if Version < 2001
         Error = -2;
         return;
       end        
%
% Some UDOP constants -----------------------------------
%
%      Data Type constant
%      -------------------
%      These constants define the type of data contains in a graph
%      The data format is 8 bit except for the following data type values:
%      4, 16, 17, 19, 24, 25, 30 : the data in format 16 bits
%      18 : data in format 32 bits
%
       Nb_Para_Fct = 256;                     % Number of bytes of the table of parameters associated to a channel

       Ofs_Para_Table = 548;  % offset in file of the Parameters table
       Ofs_Data = 31268;  % offset in file of the first data value


% -------------------------------------------------------------------------
%      Check if the channel is known in the file
%
       fseek(fid,Ofs_Para_Table,'bof');  % point to the parameters table of channel 1
       Parameters = fread(fid, Nb_Para_Fct,'int32');

       % Determine the instrument (detection based on the frequency of the AD converter)
       Freq_AD = mod(uint32(Parameters(30)),256)
       if Freq_AD ~= 3
         Dop_Model = 3010
       else
         Dop_Model = 4000;
       end;

       Mpx_Mode = uint32(Parameters(53));

       if bitand(Mpx_Mode, 2) > 0

         % if the multiplexer was active ....
         %% verify that the selected channel is included in the mpx sequence
         if Dop_Model == 4000
           if Channel > 4
             Error = -4;
             return;
           end
         end
         jt = 1;
         jt = bitshift(jt, Channel+3);
         if bitand(Mpx_Mode, jt) == 0
           Error = -4;
           return;
         end;

       else

%        Verify that the selected channel exists (non multiplex mode)
         if bitand(Mpx_Mode, 4) > 0
           % if in UDV 2D/3D mode, the channel number = 10
           Channel = 10;
         else
           fseek(fid, Ofs_Data, 'bof');
           if Get_No_Channel(fid) ~= Channel
             Error = -4;
             return;
           end;  
         end;
       end;    
%       
% Define the parameters for the selected channel
% 
       fseek(fid,Ofs_Para_Table + ((4*Nb_Para_Fct)*(Channel-1)),'bof'); 
       Parameters = fread(fid, Nb_Para_Fct,'int32');
       Error = 0
%  
end
% ----------------------------------------------------------------------
function No_Channel = Get_No_Channel(fid)
%
% Description:    Return the number of the channel attached to the data
% -----------     profile.

       N = fread(fid, 1,'uint16');          % N = Nb nytes in profile
       fseek(fid, N-5, 'cof');
       No_Channel = fread(fid, 1,'uint8');
       fseek(fid, -(N-2), 'cof');
end

% ------------------------------------------------------------------------