% ************************************************************************
function [X, Y, Info, Np, Nb_Gates] = Get_Dop_Profiles(fid, Graph, Channel, Block)
%
% Version: 3.2
% ------------
% Correction applied to version 3.1
%   can get values from Mpx simultaneous mode 
%
%
% Description:    This function extracts from an UDOP binary data file all
% -----------     the data profiles contained in single graph for a specific
%                 channel and a specific block.
%
%                 The function checks if the handle associated to the data
%                 file points to a correct UDOP binary data file. The
%                 output variable Np contains the error code.
%
%                 UDOP version supported: From 2.00.1
%
% Input:          fid   : a handle to an already opened binary data file
%                 Graph : number of the graph (1,2,3,..)
%                         each profile may contain more than one graph, for
%                         instance, the velocity profile and the echo
%                         profile.
%                 Channel: Number of the channel
%                          for DOP3000 must be set to 1
%                          In case of UDV 2D or 3D not relevant (set to 10) 
%                 Block  : number of data block
%
%
% Output:         X  : a 1 dimension data table that gives the values of
%                      the X abscissa for the selected graph. These
%                      values are depths in mm, but can be velocities in
%                      mm/s in case of the power spectrum of a single gate
%
%                 Y  : a 2 dimensions data table that contains the values of
%                      the measured data. Each row is associated to a profile
%                      and each column is associated to a gate.
%
%                      In case of raw data acquisition only:
%                      row:           data
%                       1     first emission, I signal
%                       2     first emission, Q signal
%                       3     second emission, I signal
%                       4     second emission, Q signal
%                                     etc....
%
%                 Np : if > 0  number of available data profiles
%                              (= number of rows of the Y table,
%                               or number of emissions in case of
%                               raw data acquisition)
%                      if < 0  error code
%                          -1 : not an UDOP binary data file
%                          -2 : unsupported software version
%                          -3 : the selected graphs does not exists
%                          -4 : the selected channel does not exists
%                          -5 : the selected block does not exists
%
%              Gates : Number of gates contained in the profile
%
%               Info : a 2 dimension data table that gives the additionnal
%                      information associated to each profile. The number
%                      of rows equal the number of rows of the Y table
%
%                       Column 1 : the time stamp in ms
%                                  (in micro seonds in case of raw data
%                                  acquisition)
%                       Column 2 : number of the block
%                       Column 3 : no meaning (available for future developement)
%                       Column 4 : no meaning (available for fnnuture developement)
%                       Column 5 : no meaning (available for future developement)
%                       Column 6 : number of the channel
%
% Structure of a binary profile
% -----------------------------
% A : 1 word that gives the total number of bytes contained in the profile
% B : 1 word that gives the number of bytes contained in the follwing graph
%     if 0 no more graph in the profile
% C : 1 byte that identify the type of data type contained in the graph
%     see table "Data Type constant" line 115 in this file
% D : the data bytes of the graph
%     if each data are in byte format B = number of bytes contained in the
%     graph
% Struture B,C,D repeat until B = 0
% E : 1 word = 0 indicates that no more exists
% F : 1 doubleword = time stamps in ms*10
% G : 1 word = number of the block
% H : 1 byte reserved for further use
% I : 1 byte reserved for further use
% J : 1 byte reserved for further use
% K : 1 word that gives the total number of bytes contained in the profile
%     (K = A)
%
% Date: 21 July 2016
% Author : J.-Cl Willemetz, Signal Processing SA
%          For any questions or comments please email to:
%          contact@signal-processing.com
% -------------------------------------------------------------------------
       frewind(fid);
%
% Check if the file comes from UDOP software
%
       File_Ident = fread(fid, 7,'*char');
       File_Ident = transpose(File_Ident);
       if strcmp(File_Ident, 'BINUDOP') == 0
         if strcmp(File_Ident, 'BIQUDOP') == 0
           Error = -1;
           return;
         end
       end   
%       
% Get the software version of file
%
       S = fread(fid, 1,'uint8');
       V = fread(fid, 6,'char');
       V1 = str2num(char(V(1)));
       V2 = 10*str2num(char(V(3))) + str2num(char(V(4)));      
       V3 = str2num(char(V(6)));
       Version = 1000*V1 + 10*V2 + V3;    
       if Version < 2001
         Np = -2;
         return;
       end        
%
% Some UDOP constants -----------------------------------
%
%      Data Type constant
%      -------------------
%      These constants define the type of data contains in a graph
%      The data format is 8 bit except for the following data type values:
%      4, 16, 17, 19, 24, 25, 30 : the data in format 16 bits
%      18 : data in format 32 bits
%
       Data_Type_Velocity = 0;
       Data_Type_Echo = 1;
       Data_Type_Energy = 2;
       Data_Type_Gate_FFT = 3;
       Data_Type_Phase_deg_10 = 4;           % phase in Deg*10, format 16 bits 
       Data_Type_Vitson = 5;
       Data_Type_Frequency = 6;
       Data_Type_Frequency_TR1 = 7;
       Data_Type_Frequency_TR2 = 8;
       Data_Type_Frequency_TR3 = 9;
       Data_Type_Echo_TR1 = 10;
       Data_Type_Echo_TR2 = 11;
       Data_Type_Echo_TR3 = 12;
       Data_Type_Energy_TR1 = 13;
       Data_Type_Energy_TR2 = 14;
       Data_Type_Energy_TR3 = 15;
       Data_Type_Velocity_mm_s_10 = 16;       % Velocity in mm/s * 10
       Data_Type_Velocity_mm_s = 17;          % Velocity in mm/s
       Data_Type_Flow_ml_min = 18;            % Flowrate in ml/min,  format 32 bits
       Data_Type_Diameter_10 = 19;            % diameter in mm*10, format 16 bits
       Data_Type_Aliasing_Refer = 20;         % reference profil base on 2 PRF
       Data_Type_Aliasing_Refer_TR1 = 21;     % reference profil base on 2 PRF for TR1
       Data_Type_Aliasing_Refer_TR2 = 22;     % reference profil base on 2 PRF for TR2
       Data_Type_Aliasing_Refer_TR3 = 23;     % reference profil base on 2 PRF for TR3
       Data_Type_Depth_mm_10 = 25;            % depths in [mm*10], 16 bits
       Data_Type_TGC = 28;
       Data_Type_IQ = 29;
       Data_Type_Frequency_Hz_10 = 24;
       Data_Type_Depth_mm_10 = 25;            % depths [mm*10]
       Data_Type_Angle_Deg_10 = 30;           % angle [deg*10], format 16 bits       
% ------------------------

       Nb_Para_Fct = 256;                     % Number of bytes of the table of parameters associated to a channel
       FFT_Scale_Pmin = -50;                   % FFT minimum level [dBm]
       
       Retard_Flt_50k = 10500;                 % IQ filter delay [ns]
       Retard_Flt_100k = 6300;
       Retard_Flt_150k = 3500;
       Retard_Flt_200k = 2800;
       Retard_Flt_250k = 2100;
       Retard_Flt_300k = 1800;

       Retard_Analog_DOP4000 = 1200;
       
       Ofs_Data = 31268;  % offset in file of the first data value

       Nb_Gates = 0;    
       X = [];
       Y = [];
       Info =[];
% -------------------------------------------------------------------------
%      Check if the channel is known in the file
%
       fseek(fid,548,'bof');  % point to the parameters table of channel 1
       Parameters = fread(fid, Nb_Para_Fct,'int32');

       % Determine the instrument (detection based on the frequency of the AD converter)
       Freq_AD = mod(uint32(Parameters(30)),256)
       if Freq_AD ~= 3
         Dop_Model = 3010
       else
         Dop_Model = 4000;
       end;

       Mpx_Mode = uint32(Parameters(53));

       if bitand(Mpx_Mode, 2) > 0

         % if the multiplexer was active ....
         %% verify that the selected channel is included in the mpx sequence
         if Dop_Model == 4000
           if Channel > 4
             Np = -4;
             return;
           end
         end
         jt = 1;
         jt = bitshift(jt, Channel+3);
         if bitand(Mpx_Mode, jt) == 0
           Np = -4;
           return;
         end;

       else

%        Verify that the selected channel exists (non multiplex mode)
         if bitand(Mpx_Mode, 4) > 0
           % if in UDV 2D/3D mode, the channel number = 10
           Channel = 10;
         else
           fseek(fid, Ofs_Data, 'bof');
           if Get_No_Channel(fid) ~= Channel
             Np = -4;
             return;
           end;  
         end;
       end;    
%       
% Define the parameters for the selected channel
% 
%      Select the first profile corresponding to the selected channel
       fseek(fid,548 + 1024*(Channel-1),'bof'); 
       Parameters = fread(fid, Nb_Para_Fct,'int32');
%  
       Freq_Emi = Parameters(1);     % Emitting frequency in kHZ    
       Prf_Periode_us = Parameters(6); % PRF in micro seconds
       Indice_First_Gate = Parameters(10);
       Skip_Gate = Parameters(11); % time between gates indice
       Nb_Gates = Parameters(14); %   
       Nprf = Parameters(15); %       
       Scale_Angle = Parameters(16); % velocity scaling factor       
       Sound_Speed = Parameters(20); % sound velocity in m/s
       Doppler_Angle = Parameters(21); % Doppler angle in degrees
       Module_Scale = Parameters(22);     % Emitting frequency in kHZ       
       Velocity_Offset = Parameters(23); % Coded value of the velocity Offset
       No_Flt_ispPAC = Parameters(28);   % IQ filter indice       
       Freq_AD_Info    = Parameters(30); % fixe the AD frequency 
       FFT_Scale_Pmax = Parameters(31);
       Mpx_Mode = uint32(Parameters(53));
       Freq_Latch_Tgc_kHz = Parameters(41) / 1000.0;
       UDOP_Fct_Mode =  Parameters(44)
       Cursor_On_Gate = Parameters(61);
       
       Tr1_Frequency_Offset = Parameters(71);
       Tr2_Frequency_Offset = Parameters(72);
       Tr3_Frequency_Offset = Parameters(73);       
       MD_Scale_Angle_Tr1 = Parameters(74);
       MD_Scale_Angle_Tr2 = Parameters(75);
       MD_Scale_Angle_Tr3 = Parameters(76);       
       MD_Module_Scale_Tr1 = Parameters(77);     % Emitting frequency in kHZ
       MD_Module_Scale_Tr2 = Parameters(78);     % Emitting frequency in kHZ
       MD_Module_Scale_Tr3 = Parameters(79);     % Emitting frequency in kHZ

       Acqui_IQ_Nprf = Parameters(91);
       Acqui_IQ_Nb_Gates = Parameters(92);
       Acqui_IQ_First_Gate = Parameters(93);
       
       Over_All_Gain_dB = Parameters(94);        % gain for FFT  
       Flow_unit = Parameters(112);              % 0 : ml/min
                                                 % 1 : ml/s
                                                 % 2 : dl/s                                                      
       Flow_Scale = Parameters(113);       
       Aliasing_Ctrl_1 = Parameters(114);
       Aliasing_Ctrl_2 = Parameters(115);

       Sound_Speed_Wall = Parameters(116);
       Length_Wall = Parameters(117);
       Sound_Speed_Couplant = Parameters(118);
       Length_Couplant = Parameters(119);
       Div_Freq_AD = Parameters(121);
%
% ------------------------------------------------------------------------- 
% Compute some usefull variables
%

       if Dop_Model == 4000
         Retard_Flt_IQ_ns = 1200
         Freq_AD = 1000 * 400 /(2*Div_Freq_AD);
       else
         switch No_Flt_ispPAC
           case 0
             Retard_Flt_IQ_ns = Retard_Flt_50k;
           case 1
             Retard_Flt_IQ_ns = Retard_Flt_100k;
           case 2
             Retard_Flt_IQ_ns = Retard_Flt_150k;
           case 3
             Retard_Flt_IQ_ns = Retard_Flt_200k;
           case 4
             Retard_Flt_IQ_ns = Retard_Flt_250k;
           case 5
             Retard_Flt_IQ_ns = Retard_Flt_300k;
         end;
         switch mod(Freq_AD_Info,256)
           case 0
             Freq_AD = 6000;
            case 1
              Freq_AD = 12000;
            case 2
              Freq_AD = 40000;
         end;
       end;

       Cos_AngDop = cos(Doppler_Angle*pi/180);
       Coded_Value_To_Frequency = 1000 * Scale_Angle/(256*pi*Prf_Periode_us);
       Coded_Value_To_Velocity = 128 * Coded_Value_To_Frequency * Sound_Speed /(256*Freq_Emi * Cos_AngDop);
       Coded_Value_To_Frequency_For_Tr1 = 1000 * MD_Scale_Angle_Tr1 /(256*pi*Prf_Periode_us);
       Coded_Value_To_Frequency_For_Tr2 = 1000 * MD_Scale_Angle_Tr2 /(256*pi*Prf_Periode_us);
       Coded_Value_To_Frequency_For_Tr3 = 1000 * MD_Scale_Angle_Tr3 /(256*pi*Prf_Periode_us);

       if bitand(UDOP_Fct_Mode, 2) > 0
         if bitand(UDOP_Fct_Mode, 1) > 0
           Coupling_Time_us = 1000 * ((Length_Couplant*0.1/Sound_Speed_Couplant) + (Length_Wall*0.1/Sound_Speed_Wall))
         else
           Coupling_Time_us = 1000 * ((Length_Couplant*0.2/Sound_Speed_Couplant) + (Length_Wall*0.2/Sound_Speed_Wall))
         end
       else
         Coupling_Time_us = 0.0;
       end
       
       Number_Of_Mpx_Used_Channels = Get_Number_Of_Mpx_Used_Channels(Mpx_Mode)
%
% -------------------------------------------------------------------------
% Get or define the measurement depths
%
       Ofs = Ofs_Data;
       fseek(fid, Ofs_Data, 'bof');
       if Version >= 3001

%        Point to the selected channel
%        The first profiles contain the depths of the gates. These are pseudo profiles
%        as they do not contain measured data. It exists one pseudo profile for
%        each curves contained in a profile data line
%
         while Get_No_Channel(fid) ~= Channel
           N = fread(fid, 1,'uint16');
           Ofs = Ofs + N;
           fseek(fid, N-2, 'cof');
         end;

%        Point to the selected graph
%        Each profile can contain many graphs. The data contained in each graph starts by
%        1 word that gives the number of data bytes followed by one byte that
%        gives the type of measured data (see data type constant above)

         Nb_Bytes_In_Profile = fread(fid, 1,'uint16');
         Gr = 1;
         while Gr < Graph
           Gr = Gr + 1;
           Nb_Bytes_In_Graph = fread(fid, 1,'uint16');
           if Nb_Bytes_In_Graph == 0
             Np = -3;
             return;
           end
           fseek(fid, Nb_Bytes_In_Graph + 1, 'cof');
         end
         Nb_Bytes_In_Graph = fread(fid, 1,'uint16');
         if Nb_Bytes_In_Graph == 0
           Np = -3;
           return;
         end
         Data_Type = fread(fid, 1,'uint8');
         i = Nb_Bytes_In_Graph/2;          % assume 16 bit data format
         switch Data_Type
           case Data_Type_Velocity_mm_s    % in case of gate_FFT
             V = fread(fid, i,'int16');
           case Data_Type_Diameter_10      % diameter for flow rate
             i = 1;
             V = fread(fid, i,'int32');
           case Data_Type_Depth_mm_10      % dephs in mm*10
             V = fread(fid, i,'uint16');
             V = V * 0.1;
           otherwise
             V = fread(fid, i,'uint16');  % depths values in mm
         end
         X = transpose(V);
        
%        Determine the position in the file of the first value profile
         fseek(fid, Ofs_Data, 'bof');
         Ofs_First_Value = Ofs_Data;
         if bitand(Mpx_Mode, 2) ~= 0  
           % en Mpx actif ....
           for i=1:Number_Of_Mpx_Used_Channels
             N = fread(fid, 1,'uint16');
             Ofs_First_Value = Ofs_First_Value + N;
             fseek(fid, N-2, 'cof');
           end
         else
           N = fread(fid, 1,'uint16');
           Ofs_First_Value = Ofs_First_Value + N
         end
       else

         % Compute the depts of the gates
         if Dop_Model == 4000
           for i=1:Nb_Gates
             X(i) = (Indice_First_Gate + (i-1)*(Skip_Gate+1))/(2*Freq_AD);
             X(i) = (X(i) - (Retard_Flt_IQ_ns/2000000.0)) * Sound_Speed;
           end
         else
           for i=1:Nb_Gates
             X(i) = No_Gate_To_Depth_mm (i)
           end
         end
         Ofs_First_Value = Ofs_Data
       end
%
% ----------------------------------------------------------------------
%
       fseek(fid, -2, 'eof');
       N = fread(fid, 1,'uint16');
       Ofs_Last_Profile = ftell(fid) - N;
       Ofs = Ofs_First_Value;

%      Select the first profile corresponding to the selected block.
%      Each profile contains additionnal byte. One of them is the block number
%
       fseek(fid, Ofs, 'bof'); % point to the beginning of the data profiles
       while Get_No_Block(fid) ~= Block
         if ftell(fid) >= Ofs_Last_Profile
           Np = -5;
           return;
         end;
         N = fread(fid, 1,'uint16');
         fseek(fid, N-2, 'cof');
       end;

%      define the last profile inside the selected block

       Ofs = ftell(fid);
       while Get_No_Block(fid) == Block
         if ftell(fid) == Ofs_Last_Profile
           break;
         end;
         N = fread(fid, 1,'uint16');
         fseek(fid, N-2, 'cof');
       end;
       if ftell(fid) < Ofs_Last_Profile
         Ofs_Last_Profile = ftell(fid) - N;
       end
       fseek(fid, Ofs, 'bof');

%      Select the first profile corresponding to the selected channel

       while Get_No_Channel(fid) ~= Channel
         N = fread(fid, 1,'uint16');
         fseek(fid, N-2, 'cof');
       end;

       Np = 0;
       Info = [];
       V = [];
       Ofs = ftell(fid);
       Exit_Now = 0;
       while Get_No_Channel(fid) == Channel
         Np = Np + 1;
         Nb_Bytes_In_Profile = fread(fid, 1,'uint16');
         Gr = 1;    
         while Gr < Graph
           Gr = Gr + 1;
           Nb_Bytes_In_Graph = fread(fid, 1,'uint16');
           if Nb_Bytes_In_Graph == 0
             Np = -3;
             return;
           end
           fseek(fid, Nb_Bytes_In_Graph+1, 'cof'); % point the next graph
         end
         Nb_Bytes_In_Graph = fread(fid, 1,'uint16');
         Data_Type = fread(fid, 1,'uint8');
         V = fread(fid, Nb_Bytes_In_Graph,'uint8');
         V = transpose(V);
         if Data_Type == Data_Type_IQ
           j = 1;
           for i=1:Nb_Bytes_In_Graph/2
             Y(Np,i) = int32(V(j+1)*256 + V(j));
             if Y(Np,i) > 32767
               Y(Np,i) = Y(Np,i) - 65536;
             end
             j = j + 2;
           end
           % Read the additionnal bytes
           Nb_Bytes_In_Graph = fread(fid, 1,'uint16');
           Info(1,1) = Acqui_IQ_Nprf * Prf_Periode_us;  % Time stamp in us;
           Info(1,2) = fread(fid, 1,'uint16');  % Block number
           Info(1,3) = fread(fid, 1,'uint8');   % Mark byte
           Info(1,4) = fread(fid, 1,'uint8');   % Free
           Info(1,5) = fread(fid, 1,'uint8');   % Free
           Info(1,6) = fread(fid, 1,'uint8');   % Number of channel
           Np = Acqui_IQ_Nprf
           break
         else
           switch Data_Type
             case Data_Type_Velocity
               V = Offset_Correction(V, Velocity_Offset, Nb_Gates);
               Y(Np,:) = V * Coded_Value_To_Velocity;
             case Data_Type_Echo
               m = Module_Scale / 256;
               Y(Np,:) = V * m;
             case Data_Type_Energy
               m = Module_Scale / 256;
               Y(Np,:) = m * V;
             case Data_Type_Frequency_TR1
               V = Offset_Correction(V, Tr1_Frequency_Offset, Nb_Gates);
               Y(Np,:) = V * Coded_Value_To_Frequency_For_Tr1;
             case Data_Type_Frequency_TR2
               V = Offset_Correction(V, Tr2_Frequency_Offset, Nb_Gates);
               Y(Np,:) = V * Coded_Value_To_Frequency_For_Tr2;
             case Data_Type_Frequency_TR3
               V = Offset_Correction(V, Tr3_Frequency_Offset, Nb_Gates);
               Y(Np,:) = V * Coded_Value_To_Frequency_For_Tr3;
             case Data_Type_Echo_TR1
               m = MD_Module_Scale_Tr1 / 256;
               Y(Np,:) = m * V;
             case Data_Type_Echo_TR2
               m = MD_Module_Scale_Tr2 / 256;
               Y(Np,:) = m * V;
             case Data_Type_Echo_TR3
               m = MD_Module_Scale_Tr3 / 256;
               Y(Np,:) = m * V;
             case Data_Type_Energy_TR1
               m = MD_Module_Scale_Tr1 / 256;
               Y(Np,:) = m * V;
             case Data_Type_Energy_TR2
               m = MD_Module_Scale_Tr2 / 256;
               Y(Np,:) = m * V;
             case Data_Type_Energy_TR3
               m = MD_Module_Scale_Tr3 / 256;
               Y(Np,:) = m * V;
             case Data_Type_Gate_FFT
               V = V *(FFT_Scale_Pmax - FFT_Scale_Pmin)/255.0 + FFT_Scale_Pmin;
               Y(Np,:) = V - Over_All_Gain_dB;
             case {Data_Type_Velocity_mm_s_10, Data_Type_Angle_Deg_10, Data_Type_Frequency_Hz_10}
               j = 1;
               for i=1:Nb_Gates
                 Y(Np,i) = int32(V(j+1)*256 + V(j));
                 if Y(Np,i) > 32767
                   Y(Np,i) = Y(Np,i) - 65536;
                 end
                 j = j + 2;
               end;
               Y(Np,i) = Y(Np,i) * 0.1;
             case Data_Type_Velocity_mm_s
               j = 1;
               for i=1:Nb_Gates
                 Y(Np,i) = int32(V(j+1)*256 + V(j));
                 if Y(Np,i) > 32767
                   Y(Np,i) = Y(Np,i) - 65536;
                 end
                 j = j + 2;
               end;
             case Data_Type_Flow_ml_min
               Y(Np,i) = int32(V(j+3)*16777216 + V(j+2)*65536 +  V(j+1)*256 + V(j));
           end
           
           % Read the additionnal bytes
           Nb_Bytes_In_Graph = fread(fid, 1,'uint16');
           while Nb_Bytes_In_Graph > 0
             fseek(fid, Nb_Bytes_In_Graph+1, 'cof'); % point the next graph
             Nb_Bytes_In_Graph = fread(fid, 1,'uint16');
           end
           Info(Np,1) = uint32(fread(fid, 1,'uint32')/10);  % Time stamp in ms;
           Info(Np,2) = fread(fid, 1,'uint16');  % Block number
           Info(Np,3) = fread(fid, 1,'uint8');   % Mark byte
           Info(Np,4) = fread(fid, 1,'uint8');   % Free
           Info(Np,5) = fread(fid, 1,'uint8');   % Free
           Info(Np,6) = fread(fid, 1,'uint8');   % Number of channel
         end;
         N = fread(fid, 1,'uint16');  % Nb_Bytes_In_Profile
         
%        Point to the next profile
         if bitand(Mpx_Mode, 1073741826) == 1073741826  % 0x40000002
           % Mpx simultaneous mode enabled ....
           for i=1:Number_Of_Mpx_Used_Channels-1
             N = fread(fid, 1,'uint16');
             Ofs = Ofs + N;                 % Ofs = Origin of next profile
             if Ofs > Ofs_Last_Profile
               Exit_Now = 1;
             end
             fseek(fid, Ofs, 'bof');
           end
         else
           N = fread(fid, 1,'uint16');
           Ofs = Ofs + N;                 % Ofs = Origin of next profile
           if Ofs > Ofs_Last_Profile
             Exit_Now = 1;
           end 
         end
         if Exit_Now == 1
           break;
         end
       end
end
% ----------------------------------------------------------------------
function Gate_Depth_mm = No_Gate_To_Depth_mm(Ng)
%
% Description:    Compute the depth in mm for the gate Ng
% -----------
%
% Input: Ng = number of the gate [1 ...]

       if bitand(UDOP_Fct_Mode, 1) > 0
         Gate_Depth_mm = ( (1000*(Indice_First_Gate + (Ng-1)*(Skip_Gate+1))/Freq_AD) - (Retard_analog_ns*0.001) - Coupling_Time_us) * Sound_Speed * 0.0005
       else
         Gate_Depth_mm = ( (1000*(Indice_First_Gate + (Ng-1)*(Skip_Gate+1))/Freq_AD) - (Retard_analog_ns*0.001) - Coupling_Time_us) * Sound_Speed * 0.001;
       end
end
% ----------------------------------------------------------------------
function Velocity = Offset_Correction(Value, Velocity_Offset, Nb_Gates)
%
% Description:    Correct a binary velocity value if an offset is attached
% -----------     to it.
%
% Input: binary data value read from file
%        Velocity_Offset, binary value read from the data file

       for i=1:Nb_Gates
         Value(i) = Value(i) + Velocity_Offset;
         ICOR = 0;
         if (Value(i) > 127)
           ICOR = -256;
         end
         if Value(i) < -128
           ICOR = 256;
         end
         Value(i) = Value(i) + ICOR;
         Velocity(i) = double(Value(i) - Velocity_Offset);
       end
end
% ----------------------------------------------------------------------
function No_Channel = Get_No_Channel(fid)
%
% Description:    Return the number of the channel attached to the data
% -----------     profile.

       N = fread(fid, 1,'uint16');          % N = Nb nytes in profile
       fseek(fid, N-5, 'cof');
       No_Channel = fread(fid, 1,'uint8');
       
       % Correct error in previous version....
       if No_Channel == 0
         No_Channel = 1
       end
       % end of correction....
       
       fseek(fid, -(N-2), 'cof');
end
% ----------------------------------------------------------------------
function No_Block = Get_No_Block(fid)
%
% Description:    Return the number of the block attached to the data
% -----------     profile.

       N = fread(fid, 1,'uint16');          % N = Nb nytes in profile
       fseek(fid, N-10, 'cof');
       No_Block = fread(fid, 1,'uint16');
       fseek(fid, -(N-6), 'cof');
end
% ----------------------------------------------------------------------
function Number_Of_Mpx_Used_Channels = Get_Number_Of_Mpx_Used_Channels(Mpx_Mode)
%
% Description:    Return the number of used channel
% -----------
%                 Input: Mpx_Mode
%                 Output: Number of used channels in Mpx mode actif
    Nb_Used = 0
    imax = 16384;    % 0x4000
    i = 8;
    while i ~= imax
      i = i *2;
      if bitand(Mpx_Mode, i) ~= 0
        Nb_Used = Nb_Used + 1;
      end
    end  
    Number_Of_Mpx_Used_Channels = Nb_Used;
end
% ------------------------------------------------------------------------